

#include <time.h>
#include <stdlib.h>
#include <malloc.h>
#include <math.h>
#include"rfc_amalgamated.h"
#include"lice/lice/lice.h"
#include"bass/bass.h"
#include"bass/BASS_WADSP.h"
#include"bass/tags.h"
#include"resource.h"
#include <shlobj.h>

#pragma comment(lib,"bass/bass.lib")
#pragma comment(lib,"bass/BASS_WADSP.lib")
#pragma comment(lib,"bass/tags.lib")

#define IMG_WIDTH 340
#define IMG_HEIGHT 285

// animation data
#define MAX_BALL_RADIUS 7
#define SPEED_RANGE 1.5f
#define ALPHA_DECAY 80
#define ALPHA_INC 0.08
#define PARTICLE_COUNT  70
#define MIN_SPEED 0.3f

#define LINES_COUNT 6

#define FRAME_TIME 33

#define COLOR_THEMES_COUNT 8
int colorThemes[COLOR_THEMES_COUNT][3] = { { 166, 210, 255 },
{ 215, 166, 255 },
{ 166, 255, 255 },
{ 168, 255, 166 },
{ 253, 255, 166 },
{ 255, 166, 172 },
{ 255, 193, 132 },
{ 195, 195, 195 }
};

volatile bool IsAGCEnabled=false;

int GetRandomNumber()
{
	return rand() % 255;
}

float GetRandomFloat()
{
	return GetRandomNumber() / (float)255;
}

class Particle
{
public:
	int size, fadeBack;
	float alpha, x, y, xSpeed, ySpeed, alphaDecay;

	Particle()
	{
		x = GetRandomFloat() * IMG_WIDTH;
		y = GetRandomFloat() * IMG_HEIGHT;

		alpha = GetRandomFloat();
		size = GetRandomNumber() % MAX_BALL_RADIUS;

		xSpeed = (GetRandomFloat() * (SPEED_RANGE * 2)) - SPEED_RANGE;
		ySpeed = (GetRandomFloat() * (SPEED_RANGE * 2)) - SPEED_RANGE;

		if (xSpeed>0)
			xSpeed += MIN_SPEED;
		else
			xSpeed -= MIN_SPEED;

		if (ySpeed>0)
			ySpeed += MIN_SPEED;
		else
			ySpeed -= MIN_SPEED;

		alphaDecay = GetRandomFloat() / ALPHA_DECAY;

		fadeBack = 0;
	}

	void updateParticle()
	{
		x += xSpeed;
		y += ySpeed;

		alpha -= alphaDecay;

		if (alpha <= 0)
		{
			x = GetRandomFloat() * IMG_WIDTH;
			y = GetRandomFloat() * IMG_HEIGHT;

			xSpeed = (GetRandomFloat() * (SPEED_RANGE * 2)) - SPEED_RANGE;
			ySpeed = (GetRandomFloat() * (SPEED_RANGE * 2)) - SPEED_RANGE;

			if (xSpeed>0)
				xSpeed += MIN_SPEED;
			else
				xSpeed -= MIN_SPEED;

			if (ySpeed>0)
				ySpeed += MIN_SPEED;
			else
				ySpeed -= MIN_SPEED;

			fadeBack = 1;
		}


		if (fadeBack == 1)
		{
			alpha += ALPHA_INC;

			if (alpha >= 1)
			{
				alpha = 1;
				fadeBack = 0;
			}
		}


	}
};



class NOWComponent
{
protected:
	bool compVisible;
	int compX, compY;
	int compWidth, compHeight;


public:
	NOWComponent()
	{
		compX = 0;
		compY = 0;
		compVisible = true;
		compWidth = 0;
		compHeight = 0;
	}

	virtual void SetPosition(int x, int y)
	{
		compX = x;
		compY = y;
	}

	virtual int GetPosX()
	{
		return compX;
	}

	virtual int GetPosY()
	{
		return compY;
	}

	virtual void SetSize(int width, int height)
	{
		compWidth = width;
		compHeight = height;
	}

	virtual int GetWidth()
	{
		return compWidth;
	}

	virtual int GetHeight()
	{
		return compHeight;
	}

	virtual void SetVisible(bool visible)
	{
		compVisible = visible;
	}

	virtual bool IsVisible()
	{
		return compVisible;
	}


	virtual bool IsMouseWithinArea(int x, int y)
	{
		int relX = x - compX;
		int relY = y - compY;

		if ((0<relX) && (relX <= compWidth) && (0<relY) && (relY <= compHeight))
		{
			return true;
		}
		return false;
	}

	//events!

	virtual void OnMouseLDown(int xpos, int ypos){}
	
	virtual void OnMouseRDown(int xpos, int ypos){}

	virtual void OnMouseLUp(int xpos, int ypos){}

	virtual void OnMouseRUp(int xpos, int ypos){}

	virtual void OnMouseMove(int xpos, int ypos, WPARAM fwKeys){}

	virtual void OnMouseOver(WPARAM fwKeys){}

	virtual void OnMouseLost(WPARAM fwKeys){}

	virtual void OnPaint(LICE_SysBitmap *canvas){}

	virtual ~NOWComponent(){}

};

class NOWImage : public NOWComponent
{
protected:
	LICE_IBitmap *image;

public:
	NOWImage()
	{
		image = 0;
	}

	virtual void OnPaint(LICE_SysBitmap *canvas)
	{
		LICE_Blit(canvas, image, compX, compY, 0, 0, compWidth, compHeight, 1.0, LICE_BLIT_USE_ALPHA);
	}

	virtual void SetImage(HINSTANCE hInst, int imageID)
	{
		image = LICE_LoadPNGFromResource(hInst,imageID);

		compWidth = image->getWidth();
		compHeight = image->getHeight();
	}

	virtual bool IsMouseWithinArea(int x, int y)
	{
		int relX = x - compX;
		int relY = y - compY;

		if ((0<relX) && (relX <= compWidth) && (0<relY) && (relY <= compHeight))
		{
			LICE_pixel p = LICE_GetPixel(image, relX, relY);
			if (LICE_GETA(p) > 100)
				return true;
		}
		return false;
	}

	virtual ~NOWImage()
	{
		if (image)
			delete image;
	}
};

class NOWRotatingImage : public NOWImage
{
protected:
	float angle;
	HSTREAM *hstream;

public:
	NOWRotatingImage(HSTREAM *hstream)
	{
		angle = 359;
		this->hstream = hstream;
	}

	virtual void OnPaint(LICE_SysBitmap *canvas)
	{
		if ( (*hstream != 0) && (BASS_ChannelIsActive(*hstream) == BASS_ACTIVE_PLAYING))
		{
			angle = angle - 0.05;
			if (angle < 1)
				angle = 359;
		}

		LICE_RotatedBlit(canvas, image, compX, compY, compWidth, compHeight, 0, 0, compWidth, compHeight, angle, false, 1.0f, LICE_BLIT_USE_ALPHA);
	}

	virtual ~NOWRotatingImage(){}
};

class NOWButtonListener
{
public:
	virtual void OnButtonPress(void* button) = 0;
	virtual void SetMouseFocusTo(NOWComponent *component) = 0;
};

class NOWButton : public NOWComponent
{
protected:
	LICE_IBitmap *normalImage,*downImage;
	bool isOver, isDown,lDown;
	NOWButtonListener *listener;
	bool useAlphaHitTest;
public:
	NOWButton(bool useAlphaHitTest=true)
	{
		normalImage = 0;
		downImage = 0;
		listener = 0;
		this->useAlphaHitTest = useAlphaHitTest;

		isOver = false;
		isDown = false;
		lDown = false;
	}

	virtual void SetImages(HINSTANCE hInst, int normalImageID, int downImageID)
	{
		normalImage = LICE_LoadPNGFromResource(hInst, normalImageID);
		downImage = LICE_LoadPNGFromResource(hInst, downImageID);

		compWidth = normalImage->getWidth();
		compHeight = normalImage->getHeight();
	}

	virtual void SetListener(NOWButtonListener *listener)
	{
		this->listener = listener;
	}

	virtual void OnPaint(LICE_SysBitmap *canvas)
	{
		if (isDown)
			LICE_Blit(canvas, downImage, compX, compY, 0, 0, compWidth, compHeight, 1.0, LICE_BLIT_USE_ALPHA);
		else
			LICE_Blit(canvas, normalImage, compX, compY, 0, 0, compWidth, compHeight, 1.0, LICE_BLIT_USE_ALPHA);
	}

	virtual void OnButtonClick()
	{
		if (listener)
			listener->OnButtonPress(this);
	}

	virtual bool IsMouseWithinArea(int x, int y)
	{
		int relX = x - compX;
		int relY = y - compY;
		
		if ((0<relX) && (relX <= compWidth) && (0<relY) && (relY <= compHeight))
		{
			if (useAlphaHitTest)
			{
				LICE_pixel p = LICE_GetPixel(isDown ? downImage : normalImage, relX, relY);
				if (LICE_GETA(p) > 100)
					return true;
			}
			else
			{
				return true;
			}
		}
		return false;
	}

	virtual void OnMouseMove(int xpos, int ypos, WPARAM fwKeys)
	{
		if (fwKeys == MK_LBUTTON)
		{
			if (lDown)
			{
				if (IsMouseWithinArea(xpos, ypos))
				{
					isDown = true;
				}
				else
				{
					isDown = false;
				}
			}
		}
	}

	virtual void OnMouseLDown(int xpos, int ypos)
	{
		isDown = true;
		lDown = true;
		if (listener)
			listener->SetMouseFocusTo(this);
	}

	virtual void OnMouseLUp(int xpos, int ypos)
	{
		isDown = false;
		
		if (listener)
			listener->SetMouseFocusTo(0);

		if (lDown)
		{
			if (IsMouseWithinArea(xpos, ypos))
			{
				OnButtonClick();
			}
		}
		lDown = false;
	}

	virtual ~NOWButton()
	{
		if (normalImage)
			delete normalImage;
		if (downImage)
			delete downImage;
	}

};

class NOWSliderListener
{
public:
	virtual void OnSliderChange(void* slider) = 0;
	virtual void SetMouseFocusTo(NOWComponent *component) = 0;
};

class NOWSlider : public NOWComponent
{
protected:
	int value;
	NOWSliderListener *listener;
	bool lDown;
public:
	NOWSlider()
	{
		value = 50;
		compWidth = 150;
		compHeight = 16;
		listener = 0;
		lDown = false;
	}

	virtual void SetListener(NOWSliderListener *listener)
	{
		this->listener = listener;
	}

	virtual void SetValue(int value)
	{
		if (value<0)
		{
			this->value = 0;
		}
		else if (value>100)
		{
			this->value = 100;
		}
		else
		{
			this->value = value;
		}
	}

	virtual int GetValue()
	{
		return value;
	}

	virtual void OnPaint(LICE_SysBitmap *canvas)
	{
		LICE_FillRect(canvas, compX, compY + 7, compWidth, 4, LICE_RGBA(0, 0, 0, 255));

		if (value)
			LICE_FillRect(canvas, compX, compY + 7, (int)(compWidth*(value / (float)100)), 4, LICE_RGBA(255, 255, 255, 255));
	}

	virtual void OnMouseMove(int xpos, int ypos, WPARAM fwKeys)
	{
		if (fwKeys == MK_LBUTTON)
		{
			if (lDown)
			{
				int relX = xpos - compX;

				SetValue((relX / (float)compWidth) * 100);
				if (listener)
					listener->OnSliderChange(this);
			}
		}
	}

	virtual void OnMouseLDown(int xpos, int ypos)
	{
		lDown = true;

		if (listener)
			listener->SetMouseFocusTo(this);

		int relX = xpos - compX;
		SetValue((relX / (float)compWidth) * 100);
		if (listener)
			listener->OnSliderChange(this);
	}

	virtual void OnMouseLUp(int xpos, int ypos)
	{
		lDown = false;

		if (listener)
			listener->SetMouseFocusTo(0);
	}

	virtual ~NOWSlider(){}
};

class NOWGhostSlider : public NOWSlider
{
protected:
	int ghostValue;
public:
	NOWGhostSlider()
	{
		value = 0;
		ghostValue = 0;
	}

	virtual void SetGhostValue(int value)
	{
		if (value<0)
		{
			this->ghostValue = 0;
		}
		else if (value>100)
		{
			this->ghostValue = 100;
		}
		else
		{
			this->ghostValue = value;
		}
	}

	virtual void OnMouseLUp(int xpos, int ypos)
	{
		if (lDown)
		{
			SetValue(ghostValue);
			ghostValue = 0;
			if (listener)
				listener->OnSliderChange(this);
		}

		lDown = false;

		if (listener)
			listener->SetMouseFocusTo(0);
	}

	virtual void OnMouseLDown(int xpos, int ypos)
	{
		lDown = true;

		if (listener)
			listener->SetMouseFocusTo(this);

		int relX = xpos - compX;
		SetGhostValue((relX / (float)compWidth) * 100);
	}

	virtual void OnMouseMove(int xpos, int ypos, WPARAM fwKeys)
	{
		if (fwKeys == MK_LBUTTON)
		{
			if (lDown)
			{
				int relX = xpos - compX;
				SetGhostValue((relX / (float)compWidth) * 100);
			}
		}
	}

	virtual void OnPaint(LICE_SysBitmap *canvas)
	{
		LICE_FillRect(canvas, compX, compY + 7, compWidth, 4, LICE_RGBA(0, 0, 0, 255));

		if (value)
			LICE_FillRect(canvas, compX, compY + 7, (int)(compWidth*(value / (float)100)), 4, LICE_RGBA(255, 255, 255, 255));

		if (ghostValue)
			LICE_FillRect(canvas, compX, compY + 7, (int)(compWidth*(ghostValue / (float)100)), 4, LICE_RGBA(192, 192, 192, 192), 0.6f);
	}

	virtual ~NOWGhostSlider(){}
};

class NOWToggleButton : public NOWButton
{
protected:
	bool toggled;
public:
	NOWToggleButton() :NOWButton(false)
	{
		toggled = false;
	}

	virtual bool IsToggled()
	{
		return toggled;
	}

	virtual void SetToggleState(bool state)
	{
		toggled = state;
	}

	virtual void OnPaint(LICE_SysBitmap *canvas)
	{
		bool showtoggle = isDown ? !toggled : toggled;

		if (showtoggle)
			LICE_Blit(canvas, downImage, compX, compY, 0, 0, compWidth, compHeight, 1.0, LICE_BLIT_USE_ALPHA);
		else
			LICE_Blit(canvas, normalImage, compX, compY, 0, 0, compWidth, compHeight, 1.0, LICE_BLIT_USE_ALPHA);
	}

	virtual void OnButtonClick()
	{
		toggled = !toggled;

		if (listener)
			listener->OnButtonPress(this);
	}

	virtual ~NOWToggleButton(){}
};

class NOWCheckBox : public NOWToggleButton
{
protected:
	KString text;
	KFont *font;
public:
	NOWCheckBox(KFont *font)
	{
		this->font = font;

		compWidth = 200;
		compHeight = 16;
	}

	static RECT CalculateTextSize(const wchar_t *text, HFONT hFont)
	{
		HDC hDC = CreateICW(L"DISPLAY", NULL, NULL, NULL);
		HGDIOBJ hOldFont = SelectObject(hDC, hFont);
		RECT sz = { 0, 0, 0, 0 };
		DrawTextW(hDC, text, lstrlenW(text), &sz, DT_CALCRECT | DT_NOPREFIX);
		SelectObject(hDC, hOldFont);
		DeleteDC(hDC);
		return sz;
	}

	virtual void SetText(KString text)
	{
		this->text = text;

		if (normalImage)
		{
			RECT rect = CalculateTextSize((const wchar_t*)text, font->GetFontHandle());
			compWidth = normalImage->getWidth() + 5 + rect.right;
		}
	}

	virtual void SetImages(HINSTANCE hInst, int normalImageID, int downImageID)
	{
		normalImage = LICE_LoadPNGFromResource(hInst, normalImageID);
		downImage = LICE_LoadPNGFromResource(hInst, downImageID);

		compHeight = normalImage->getHeight();
	}

	virtual void OnPaint(LICE_SysBitmap *canvas)
	{
		NOWToggleButton::OnPaint(canvas);

		HDC hdc = canvas->getDC();
		int prevBkMode = SetBkMode(hdc, TRANSPARENT);
		COLORREF prevCol = SetTextColor(hdc, RGB(0, 0, 0));
		HGDIOBJ oldFont = SelectObject(hdc, font->GetFontHandle());

		RECT rect;
		rect.left = compX + normalImage->getWidth()+5;
		rect.top = compY;
		rect.right = compX + compWidth;
		rect.bottom = compY + compHeight;
		DrawTextW(hdc, (const wchar_t*)text, text.GetLength(), &rect, DT_LEFT | DT_NOPREFIX | DT_END_ELLIPSIS | DT_SINGLELINE | DT_VCENTER);

		SetTextColor(hdc, prevCol);
		SetBkMode(hdc, prevBkMode);
		SelectObject(hdc, oldFont);
	}

	virtual ~NOWCheckBox(){}
};

class NOWLabel : public NOWComponent
{
protected:
	KString text;
	KFont *font;
public:
	NOWLabel(KFont *font)
	{
		this->font = font;

		compWidth = 150;
		compHeight = 20;
	}

	virtual void SetText(KString text)
	{
		if (text.GetLength() == 0)
			this->text = L"Unknown";
		else
			this->text = text;
	}

	virtual bool IsMouseWithinArea(int x, int y)
	{
		return false;
	}

	virtual void OnPaint(LICE_SysBitmap *canvas)
	{
		HDC hdc = canvas->getDC();
		int prevBkMode = SetBkMode(hdc, TRANSPARENT);
		COLORREF prevCol = SetTextColor(hdc, RGB(0,0,0));
		HGDIOBJ oldFont = SelectObject(hdc, font->GetFontHandle());

		RECT rect;
		rect.left = compX;
		rect.top = compY;
		rect.right = compX + compWidth;
		rect.bottom = compY + compHeight;
		DrawTextW(hdc, (const wchar_t*)text, text.GetLength(), &rect, DT_LEFT | DT_NOPREFIX|DT_END_ELLIPSIS|DT_SINGLELINE|DT_VCENTER);

		SetTextColor(hdc, prevCol);
		SetBkMode(hdc, prevBkMode);
		SelectObject(hdc, oldFont);
	}

	virtual ~NOWLabel(){}
};

#define TEXT_LINES_COUNT 128
static const wchar_t *textLines[TEXT_LINES_COUNT] = { 
	L"Simplicity is the ultimate sophistication.    ",
	L"The man who learns the art of sharing is the richest man in the world.He may be poor, but his inner being is rich.    ",
	L"Sometimes you have to stand alone to prove that you can still stand.    ",
	L"Never let an old flame burn you twice.    ",
	L"The most beautiful music in the world is your own heartbeat It gives an assurance that you'll survive even when the whole world leaves you alone...    ",
	L"All birds find shelter during a rain. But eagle avoids rain by flying above the clouds.    ",
	L"A loving heart is the beginning of all knowledge.    ",
	L"Beauty isn't about having a pretty face. it's about having a pretty mind, a pretty heart and a pretty soul.    ",
	L"People are like stained - glass windows.They sparkle and shine when the sun is out, but when the darkness sets in their true beauty is revealed only if there is light from within.    ",
	L"If you know where you are going, you are not learning.    ",
	L"When I was a kid my mom would send me off to school each day with the words, Remember: be happy! The most important thing today is that you are happy!    ",
	L"Let your imagination takes you to where you wish to be.    ",
	L"Explore your own innermost thoughts to create content that will evoke deeply relatable emotions and passion in others.    ",
	L"If you have few months to live, what will you do?    ",
	L"A man is not judged by how much he made or the power he gained in his life, but by the love that he shared and the lives that he touched.    ",
	L"A real love is not having your beloved in bed but in your dreams and thoughts.    ",
	L"We are never the same person twice.    ",
	L"Be yourself, because really, who else can you be?    ",
	L"Beauty is a state of mind, not a state of the body.    ",
	L"What other people think of you is none of your business.    ",
	L"Stop taking life so seriously. You'll never make it out alive.    ",
	L"When everything is lonely I can be my best friend.    ",
	L"The loneliest moment in someone's life is when they are watching their whole world fall apart, and all they can do is stare blankly.    ",
	L"Online communities are an expression of loneliness.    ",
	L"If you don't know where you're going, any road'll take you there.    ",
	L"The most important thing is to enjoy your life.    ",
	L"Some people have lives, some people have music.    ",
	L"This is your life and its ending one moment at a time.    ",
	L"Don't take life too seriously. Punch it in the face when it needs a good hit. Laugh at it.    ",
	L"The past has no power over the present moment.    ",
	L"If you end up with a boring miserable life because you listened to your mom, your dad, your teacher, your priest, or some guy on television telling you how to do your shit, then you deserve it.    ",
	L"How nice - to feel nothing, and still get full credit for being alive.    ",
	L"Life doesn't imitate art, it imitates bad television.    ",
	L"Maybe this world is another planet's hell.    ",
	L"In the name of God, stop a moment, cease your work, look around you.    ",
	L"If you want to make good laugh, tell them about your plans.    ",
	L"Many people die at twenty five and aren't buried until they are seventy five.    ",
	L"But how could you live and have no story to tell?    ",
	L"Winners lose more than losers. Because they stay in the game.    ",
	L"I didn't want to wake up. I was having a much better time asleep. And that's really sad.It was almost like a reverse nightmare, like when you wake up from a nightmare you're so relieved. I woke up into a nightmare.    ",
	L"Now friendships depend on and are subject to the Facebook's privacy policy.    ",
	L"People who are not on Facebook are face to face.    ",
	L"And I have seen people on Facebook with their real names and own pictures but fake personalities.    ",
	L"The greatest people and most successful ones are the ones who never went to school.    ",
	L"Internet makes you think that you have millions of friends, whereas you are your own friend.    ",
	L"Facebook gives people an illusory sense of being LIKED.    ",
	L"You think Facebook was created by some asshole at Harvard ? That asshole is on the government payroll.    ",
	L"Facebook is the only place where it's acceptable to talk to a wall.    ",
	L"Instead of Facebook asking Whats on your mind? , it should probably ask WTF is YOUR problem?    ",
	L"Without music, life would be a mistake.    ",
	L"One good thing about music, when it hits you, you feel no pain.    ",
	L"Music expresses that which cannot be put into words and that which cannot remain silent.    ",
	L"People haven't always been there for me but music always has.    ",
	L"Everything in the universe has a rhythm, everything dances.    ",
	L"There ain't no devil, only God when he's drunk.    ",
	L"Music is the strongest form of magic.    ",
	L"Music acts like a magic key, to which the most tightly closed heart opens.    ",
	L"Tell me what you listen to, and I'll tell you who you are.    ",
	L"Computers are finite machines; when given the same input, they always produce the same output.    ",
	L"The computer is incredibly fast, accurate, and stupid. Man is unbelievably slow, inaccurate, and brilliant.    ",
	L"Sometimes... dreams are all that separate us from the machines.    ",
	L"Computers make it easier to do a lot of things but most of the things they make it easier to do don't need to be done.    ",
	L"The brain may take advice, but not the heart.    ",
	L"The wisest man is he who does not require advice.    ",
	L"The trouble with advice is that it's usually something you don't want to hear.    ",
	L"Why is art beautiful ? Because it's useless.    ",
	L"Art is a marriage of the conscious and the unconscious.    ",
	L"All thinking men are atheists.    ",
	L"Everything has its beauty, but not everyone sees it.    ",
	L"God loved the birds and invented trees. Man loved the birds and invented cages.    ",
	L"No one ever reads a book.He reads himself through books.    ",
	L"Old men can make war, but it is children who will make history.    ",
	L"To lose a child ... was something that could end one's world. One could never get back to how it was before. The stars went out. The moon disappeared. The birds became silent.    ",
	L"The soul is healed by being with children.    ",
	L"That's the real trouble with the world, too many people grow up.    ",
	L"Children keep a family together, especially when one can't get a babysitter.    ",
	L"If you carry your childhood with you, you never become older.    ",
	L"The secret to creativity is knowing how to hide your sources.    ",
	L"You need chaos in your soul to give birth to a dancing star.    ",
	L"There are more dead people than living. And their numbers are increasing.    ",
	L"As soon as one is born, one starts dying.    ",
	L"No one on his deathbed ever said, I wish I had spent more time on my business.    ",
	L"Death aims only once, but never misses.    ",
	L"We live as we die, and die as we live.    ",
	L"My only fear of death is reincarnation.    ",
	L"When a man hurries, the devil smiles.    ",
	L"The devil has no power ... except in the dark.    ",
	L"Dreams are more real than reality itself, they're closer to the self.    ",
	L"If we couldn't dream, our lives wouldn't mean anything anymore.    ",
	L"A dream is a wish your heart makes.    ",
	L"Dreams: the place most of us get what we need.    ",
	L"A single dream is more powerful than a thousand realities.    ",
	L"Personally I am in favour of education but a university is not the place for it.    ",
	L"Education makes some men wiser, others more ridiculous and foolish!    ",
	L"All you have to do to educate a child is leave him alone and teach him to read. The rest is brainwashing.    ",
	L"Man does not control his own fate. The women in his life do that for him.    ",
	L"Fashions fade, style is eternal.    ",
	L"I don't do fashion, I AM fashion.    ",
	L"Fashion is what you adopt when you don't know who you are.    ",
	L"Flowers are the sweetest things that God ever made, and forgot to put a soul into.    ",
	L"There are always flowers for those who want to see them.    ",
	L"I talk to God but the sky is empty.    ",
	L"Government is not the solution to our problem. Government is the problem.    ",
	L"Governments have a tendency not to solve problems, only to rearrange them.    ",
	L"Imagination is more important than knowledge.    ",
	L"Information is the mortar that both builds and destroys empires.    ",
	L"The strongest man in the world is he who stands most alone.    ",
	L"Sleep is the best meditation.    ",
	L"Be master OF mind rather than mastered BY mind.    ",
	L"Be here now.Be someplace else later. Is that so complicated ?    ",
	L"Quiet the mind, and the soul will speak.    ",
	L"If you want to conquer the anxiety of life, live in the moment, live in the breath.    ",
	L"The greatest business of a man is to improve his mind.    ",
	L"One man that has a mind and knows it can always beat ten men who haven't and don't.    ",
	L"There are some things one remembers even though they may never have happened.    ",
	L"You can't run away from memories, no matter how hard you try.    ",
	L"A person's memory is everything, really. Memory is identity. It's you.    ",
	L"We do not remember days, we remember moments.    ",
	L"I think memories are like dreams.Not reliable proof of anything. I can't prove a memory any more than I can prove a dream.    ",
	L"Money doesn't spend in hell... The devil deals in a different coin.    ",
	L"When money talks, nobody cares what kind of grammar it uses.    ",
	L"Money was invented to spend.    ",
	L"The tragedy of old age is not that one is old, but that one is young.    ",
	L"The old are in a second childhood.    ",
	L"We've got to live, no matter how many skies have fallen.    ",
	L"Pain is inevitable. Suffering is optional.    ",
	L"As long as one suffers one lives.    ",
	L"Suffering is part of our training program for becoming wise    "
};

class NOWScrollLabel : public NOWLabel
{
protected:
	int rectX1, rectX2, rectX3;
	int rectWidth1, rectWidth2, rectWidth3;
	int lineIndex1, lineIndex2, lineIndex3;

	static const int BLEND_GAP = 10;
	LICE_MemBitmap *leftBlendingRect, *rightBlendingRect;
public:
	NOWScrollLabel(KFont *font) : NOWLabel(font)
	{
		compWidth = 247;
		ResetScrollText();

		leftBlendingRect = new LICE_MemBitmap(BLEND_GAP, 20);
		rightBlendingRect = new LICE_MemBitmap(BLEND_GAP, 20);
	}

	void BlendLeftRightRect(LICE_SysBitmap *canvas)
	{
		for (int j = compY; j < (compY + compHeight); j++)
		{
			for (int i = compX; i < (compX + BLEND_GAP); i++)
			{
				int relativeX = i - compX;
				LICE_pixel p = LICE_GetPixel(leftBlendingRect, relativeX, j - compY);
				LICE_PutPixel(canvas, i, j, p, 1 - (relativeX / (float)BLEND_GAP), LICE_BLIT_USE_ALPHA);
			}

			int realX = (compX + compWidth) - BLEND_GAP;
			for (int i = realX; i < (realX + BLEND_GAP); i++)
			{
				int relativeX = i - realX;
				LICE_pixel p = LICE_GetPixel(rightBlendingRect, relativeX, j - compY);
				LICE_PutPixel(canvas, i, j, p, relativeX / (float)BLEND_GAP, LICE_BLIT_USE_ALPHA);
			}
		}
	}

	virtual void ResetScrollText()
	{
		rectX1 = compX + (compWidth/2);
		lineIndex1 = rand() % TEXT_LINES_COUNT;
		rectWidth1 = NOWCheckBox::CalculateTextSize(textLines[lineIndex1], font->GetFontHandle()).right;

		rectX2 = rectX1 + rectWidth1;
		lineIndex2 = rand() % TEXT_LINES_COUNT;
		rectWidth2 = NOWCheckBox::CalculateTextSize(textLines[lineIndex2], font->GetFontHandle()).right;

		rectX3 = rectX2 + rectWidth2;
		lineIndex3 = rand() % TEXT_LINES_COUNT;
		rectWidth3 = NOWCheckBox::CalculateTextSize(textLines[lineIndex3], font->GetFontHandle()).right;
	}

	virtual void SetVisible(bool visible)
	{
		compVisible = visible;

		if (!visible)
			ResetScrollText();
	}

	virtual void OnPaint(LICE_SysBitmap *canvas)
	{
		LICE_Blit(leftBlendingRect, canvas, 0, 0, compX, compY, BLEND_GAP, compHeight, 1.0f, LICE_BLIT_MODE_COPY);
		LICE_Blit(rightBlendingRect, canvas, 0, 0, (compX + compWidth) - BLEND_GAP, compY, BLEND_GAP, compHeight, 1.0f, LICE_BLIT_MODE_COPY);

		rectX1--;
		rectX2--;
		rectX3--;

		HDC hdc = canvas->getDC();
		int prevBkMode = SetBkMode(hdc, TRANSPARENT);
		COLORREF prevCol = SetTextColor(hdc, RGB(0, 0, 0));
		HGDIOBJ oldFont = SelectObject(hdc, font->GetFontHandle());

		HRGN hrgn = CreateRectRgn(compX, compY, compX + compWidth, compY+compHeight);
		SelectClipRgn(hdc, hrgn);

		RECT rect;
		rect.top = compY;
		rect.bottom = compY + compHeight;

		rect.left = rectX1;	
		rect.right = rectX1 + rectWidth1;	
		DrawTextW(hdc, textLines[lineIndex1], wcslen(textLines[lineIndex1]), &rect, DT_LEFT | DT_VCENTER);

		rect.left = rectX2;
		rect.right = rectX2 + rectWidth2;
		DrawTextW(hdc, textLines[lineIndex2], wcslen(textLines[lineIndex2]), &rect, DT_LEFT | DT_VCENTER);

		rect.left = rectX3;
		rect.right = rectX3 + rectWidth3;
		DrawTextW(hdc, textLines[lineIndex3], wcslen(textLines[lineIndex3]), &rect, DT_LEFT | DT_VCENTER);

		SelectClipRgn(hdc, NULL);
		DeleteObject(hrgn);

		SetTextColor(hdc, prevCol);
		SetBkMode(hdc, prevBkMode);
		SelectObject(hdc, oldFont);

		BlendLeftRightRect(canvas);

		if (rectX1 < (compX - rectWidth1))
		{
			rectX1 = rectX3 + rectWidth3;
			lineIndex1 = rand() % TEXT_LINES_COUNT;
			rectWidth1 = NOWCheckBox::CalculateTextSize(textLines[lineIndex1], font->GetFontHandle()).right;
		}

		if (rectX2 < (compX - rectWidth2))
		{
			rectX2 = rectX1 + rectWidth1;
			lineIndex2 = rand() % TEXT_LINES_COUNT;
			rectWidth2 = NOWCheckBox::CalculateTextSize(textLines[lineIndex2], font->GetFontHandle()).right;
		}

		if (rectX3 < (compX - rectWidth3))
		{
			rectX3 = rectX2 + rectWidth2;
			lineIndex3 = rand() % TEXT_LINES_COUNT;
			rectWidth3 = NOWCheckBox::CalculateTextSize(textLines[lineIndex3], font->GetFontHandle()).right;
		}
	}

	virtual ~NOWScrollLabel()
	{
		delete leftBlendingRect;
		delete rightBlendingRect;
	}
};


class NOWSongTitleLabel : public NOWLabel
{
protected:
	int rectX1, rectX2;
	int rectWidth1, rectWidth2;

	wchar_t* scrollSeperator;
	volatile bool shouldScroll;

	static const int BLEND_GAP = 15;
	LICE_MemBitmap *leftBlendingRect, *rightBlendingRect;
public:
	NOWSongTitleLabel(KFont *font) : NOWLabel(font)
	{
		scrollSeperator = L"    ";
		shouldScroll = false;

		leftBlendingRect = new LICE_MemBitmap(BLEND_GAP, 20);
		rightBlendingRect = new LICE_MemBitmap(BLEND_GAP, 20);
	}

	virtual void SetText(KString text)
	{
		shouldScroll = false;

		NOWLabel::SetText(text);

		rectX1 = compX;
		rectWidth1 = NOWCheckBox::CalculateTextSize((const wchar_t*)text, font->GetFontHandle()).right;

		if ((rectWidth1 + 10) > compWidth)
		{			
			this->text = text + scrollSeperator;
			rectWidth1 = NOWCheckBox::CalculateTextSize((const wchar_t*)this->text, font->GetFontHandle()).right;

			rectX2 = rectX1 + rectWidth1;
			rectWidth2 = rectWidth1;

			shouldScroll = true;
		}
		else
		{
			shouldScroll = false;
		}
	}

	void BlendLeftRightRect(LICE_SysBitmap *canvas)
	{
		for (int j = compY; j < (compY+compHeight); j++)
		{
			for (int i = compX; i < (compX + BLEND_GAP); i++)
			{
				int relativeX = i - compX;
				LICE_pixel p = LICE_GetPixel(leftBlendingRect, relativeX, j - compY);
				LICE_PutPixel(canvas, i, j, p, 1 - (relativeX / (float)BLEND_GAP),LICE_BLIT_USE_ALPHA);
			}

			int realX = (compX + compWidth) - BLEND_GAP;
			for (int i = realX; i < (realX + BLEND_GAP); i++)
			{
				int relativeX = i - realX;
				LICE_pixel p = LICE_GetPixel(rightBlendingRect, relativeX, j - compY);
				LICE_PutPixel(canvas, i, j, p, relativeX / (float)BLEND_GAP, LICE_BLIT_USE_ALPHA);
			}
		}
	}

	virtual void OnPaint(LICE_SysBitmap *canvas)
	{
		if (!shouldScroll)
		{
			NOWLabel::OnPaint(canvas);
		}
		else
		{
			LICE_Blit(leftBlendingRect, canvas, 0, 0, compX, compY, BLEND_GAP, compHeight, 1.0f, LICE_BLIT_MODE_COPY);
			LICE_Blit(rightBlendingRect, canvas, 0, 0, (compX + compWidth) - BLEND_GAP, compY, BLEND_GAP, compHeight, 1.0f, LICE_BLIT_MODE_COPY);

			rectX1--;
			rectX2--;

			HDC hdc = canvas->getDC();
			int prevBkMode = SetBkMode(hdc, TRANSPARENT);
			COLORREF prevCol = SetTextColor(hdc, RGB(0, 0, 0));
			HGDIOBJ oldFont = SelectObject(hdc, font->GetFontHandle());

			HRGN hrgn = CreateRectRgn(compX, compY, compX + compWidth, compY + compHeight);
			SelectClipRgn(hdc, hrgn);

			RECT rect;
			rect.top = compY;
			rect.bottom = compY + compHeight;

			rect.left = rectX1;
			rect.right = rectX1 + rectWidth1;
			DrawTextW(hdc, (const wchar_t*)text, text.GetLength(), &rect, DT_LEFT | DT_VCENTER | DT_NOPREFIX);

			rect.left = rectX2;
			rect.right = rectX2 + rectWidth2;
			DrawTextW(hdc, (const wchar_t*)text, text.GetLength(), &rect, DT_LEFT | DT_VCENTER | DT_NOPREFIX);

			SelectClipRgn(hdc, NULL);
			DeleteObject(hrgn);

			SetTextColor(hdc, prevCol);
			SetBkMode(hdc, prevBkMode);
			SelectObject(hdc, oldFont);

			BlendLeftRightRect(canvas);

			if (rectX1 < (compX - rectWidth1))
			{
				rectX1 = rectX2 + rectWidth2;
			}

			if (rectX2 < (compX - rectWidth2))
			{
				rectX2 = rectX1 + rectWidth1;
			}

		}
	}

	virtual ~NOWSongTitleLabel()
	{
		delete leftBlendingRect;
		delete rightBlendingRect;
	}
};

class NOWTimeLabel : public NOWComponent
{
protected:
	char text[32];
	KFont *font;
	bool alignLeft;
public:
	NOWTimeLabel(KFont *font,bool alignLeft = true)
	{
		this->alignLeft = alignLeft;

		strcpy(text, "0:00");
		this->font = font;

		compWidth = 45;
		compHeight = 20;
	}

	virtual void SetTime(int hour,int min,int sec)
	{
		if (hour == 0)
		{
			sprintf(text, "%02d:%02d", min, sec);
		}
		else
		{
			sprintf(text, "%d:%02d:%02d", hour, min, sec);
		}
	}

	virtual bool IsMouseWithinArea(int x, int y)
	{
		return false;
	}

	virtual void OnPaint(LICE_SysBitmap *canvas)
	{
		HDC hdc = canvas->getDC();
		int prevBkMode = SetBkMode(hdc, TRANSPARENT);
		COLORREF prevCol = SetTextColor(hdc, RGB(0, 0, 0));
		HGDIOBJ oldFont = SelectObject(hdc, font->GetFontHandle());

		RECT rect;
		rect.left = compX;
		rect.top = compY;
		rect.right = compX + compWidth;
		rect.bottom = compY + compHeight;
		DrawTextA(hdc, text, strlen(text), &rect, (alignLeft?DT_LEFT:DT_RIGHT)| DT_NOPREFIX | DT_END_ELLIPSIS | DT_SINGLELINE | DT_VCENTER);

		SetTextColor(hdc, prevCol);
		SetBkMode(hdc, prevBkMode);
		SelectObject(hdc, oldFont);
	}

	virtual ~NOWTimeLabel(){}
};

class NOWPlayButtton : public NOWButton
{
protected:
	LICE_IBitmap *pauseNormalImage, *pauseDownImage;
	bool isPlaying;
public:
	NOWPlayButtton()
	{
		pauseNormalImage = 0;
		pauseDownImage = 0;
		isPlaying = false;
	}

	virtual void SetImages(HINSTANCE hInst, int normalImageID, int downImageID, int pauseNormalImageID, int pauseDownImageID)
	{
		normalImage = LICE_LoadPNGFromResource(hInst, normalImageID);
		downImage = LICE_LoadPNGFromResource(hInst, downImageID);
		pauseNormalImage = LICE_LoadPNGFromResource(hInst, pauseNormalImageID);
		pauseDownImage = LICE_LoadPNGFromResource(hInst, pauseDownImageID);

		compWidth = normalImage->getWidth();
		compHeight = normalImage->getHeight();
	}

	virtual void SetPlayingState(bool state)
	{
		isPlaying = state;
	}

	virtual bool GetPlayingState()
	{
		return isPlaying;
	}

	virtual void OnPaint(LICE_SysBitmap *canvas)
	{
		if (isDown)
			LICE_Blit(canvas, isPlaying ? pauseDownImage:downImage, compX, compY, 0, 0, compWidth, compHeight, 1.0, LICE_BLIT_USE_ALPHA);
		else
			LICE_Blit(canvas, isPlaying ? pauseNormalImage:normalImage, compX, compY, 0, 0, compWidth, compHeight, 1.0, LICE_BLIT_USE_ALPHA);
	}

	virtual void OnButtonClick()
	{
		isPlaying = !isPlaying;

		if (listener)
			listener->OnButtonPress(this);
	}

	virtual ~NOWPlayButtton()
	{
		if (pauseNormalImage)
			delete pauseNormalImage;
		if (pauseDownImage)
			delete pauseDownImage;
	}
};

class NOWImageLabel : public NOWLabel
{
protected:
	LICE_IBitmap *backImage;
public:
	NOWImageLabel(KFont *font) :NOWLabel(font)
	{

	}

	virtual void SetImage(HINSTANCE hInst, int imageID)
	{
		backImage = LICE_LoadPNGFromResource(hInst, imageID);

		compWidth = backImage->getWidth();
		compHeight = backImage->getHeight();
	}

	virtual void OnPaint(LICE_SysBitmap *canvas)
	{

		LICE_Blit(canvas, backImage, compX, compY, 0, 0, compWidth, compHeight, 1.0, LICE_BLIT_USE_ALPHA);

		HDC hdc = canvas->getDC();
		int prevBkMode = SetBkMode(hdc, TRANSPARENT);
		COLORREF prevCol = SetTextColor(hdc, RGB(30, 30, 30));
		HGDIOBJ oldFont = SelectObject(hdc, font->GetFontHandle());

		RECT rect;
		rect.left = compX;
		rect.top = compY;
		rect.right = compX + compWidth;
		rect.bottom = compY + compHeight;
		DrawTextW(hdc, (const wchar_t*)text, text.GetLength(), &rect, DT_CENTER | DT_NOPREFIX | DT_SINGLELINE | DT_VCENTER);

		SetTextColor(hdc, prevCol);
		SetBkMode(hdc, prevBkMode);
		SelectObject(hdc, oldFont);
	}

	virtual ~NOWImageLabel()
	{
		if (backImage)
			delete backImage;
	}
};

class NOWImageWithAnalyser : public NOWImage
{
protected:
	volatile bool enableVis;
	HSTREAM hstream;
public:
	NOWImageWithAnalyser()
	{
		enableVis = false;
		hstream = 0;
	}

	virtual void SetEnableVis(bool state, HSTREAM hstream)
	{
		enableVis = state;
		this->hstream = hstream;
	}

	virtual void OnPaint(LICE_SysBitmap *canvas)
	{
		LICE_Blit(canvas, image, compX, compY, 0, 0, compWidth, compHeight, 1.0, LICE_BLIT_USE_ALPHA);

		// draw vis
		#define VIS_POS_X 53
		#define VIS_POS_Y 136
		#define VIS_HEIGHT 24
		#define VIS_WIDTH 55

		if (enableVis && (hstream != 0) && (BASS_ChannelIsActive(hstream) == BASS_ACTIVE_PLAYING))
		{
			float fft[128]; // fft data buffer
			BASS_ChannelGetData(hstream, fft, BASS_DATA_FFT256 | BASS_DATA_FFT_REMOVEDC);
			for (int i = 0; i < VIS_WIDTH; i++)
			{
				int val = sqrt(fft[i]) * 3 * VIS_HEIGHT - 4; // scale it (sqrt to make low values more visible)

				if (val<0)
					val *= -1;
				if (val>VIS_HEIGHT)
					val = VIS_HEIGHT;

				LICE_Line(canvas, VIS_POS_X + i, VIS_POS_Y, VIS_POS_X + i, VIS_POS_Y - val, LICE_RGBA(255, 255, 255, 255), 0.5f);
			}
		}
	}

	virtual ~NOWImageWithAnalyser(){}
};

//=============================auto volume code===================================
const float RegainThreshold = 0.75f;
const float MUL_MIN = 1.0f;
const float MUL_MAX = 10.0f;
float mul = MUL_MAX;
// presets
float MUL_STEP = 0.9f;
float normalizeMax = 400;
template<class T> inline const T& limit(const T& val, const T& min, const T& max)
{
	if (val < min) {
		return min;
	}
	else if (val > max) {
		return max;
	}
	else {
		return val;
	}
}
void autoVolume(short *samples, int numsamples, int nch, int srate)
{
	size_t len = numsamples * nch;
	float max = 0.0f;
	for (size_t i = 0; i < len; i++) {
		float tmp = float(samples[i]);
		max = max > tmp ? max : tmp;
	}
	float upper = mul > (normalizeMax / 100.0f) ? (normalizeMax / 100.0f) : mul;

	// Evaluate an adequate 'mul' coefficient based on previous state, current samples level, etc
	if (mul > SHRT_MAX / max || mul > normalizeMax / 100.0f) {
		mul = limit((float)(SHRT_MAX / max), MUL_MIN, upper);
	}
	else {
		// here we make sure that the current sound level (max * mul) will not rise above 'RegainThreshold'.
		if (max < (SHRT_MAX / mul) * RegainThreshold) {
			// note that in one second, in average, the sum of ((float)numsamples / fmt.freq) will be 1.
			float step = MUL_STEP * ((float)numsamples / srate);
			if (mul + step <= normalizeMax / 100.0f) {
				mul += step;
			}
			else { // mul + step > cfg->normalizeMax/100.0f
				// make sure that the last increment will be performed, even if it's smaller than 'step'
				// otherwise, current amplification could be displayed as 399% instead of 400%, for example.
				mul = normalizeMax / 100.0f;
			}
		}
	}

	// Scale & clamp the samples
	int x = numsamples;
	short *a = samples;
	if (nch == 2)
	while (x--)
	{
		int l, r;

		// process l & r
		l = mul * a[0];
		r = mul * a[1];

		if (l < -32768) l = -32768;
		if (l > 32767) l = 32767;
		if (r < -32768) r = -32768;
		if (r > 32767) r = 32767;

		a[0] = l;
		a[1] = r;

		a += 2;
	}

}
//==================================================================================

void CALLBACK AGCDSPCallback(HDSP handle, DWORD channel, void *buffer, DWORD length, void *user)
{
	if (IsAGCEnabled)
	{
		autoVolume((short*)buffer, length/4, 2, 44100);
	}
}


class MainWindow : public KWindow, public KThread, public NOWButtonListener, public NOWSliderListener, public KTimerListener
{
private:
	KIcon wndIcon;
	KString applicationPath;
	wchar_t szConfPath[MAX_PATH];

	volatile bool enableDrawing;
	LICE_SysBitmap *backBuffer, *backPanel;
	LICE_IBitmap *gradImage, *ballImage, *linesImage;
	LICE_IBitmap *shapeImg, *maskImg, *glassImg;

	Particle *particles[PARTICLE_COUNT];

	int lineCurrentXPos;

	// for fps control
	double pcFreq;

	// client area drag data
	bool windowDraging;
	int cld_mouse_x;
	int cld_mouse_y;

	KPointerList<NOWComponent*> *componentList;
	NOWComponent *mouseOverComponent,*mouseFocusedComponent;

	NOWButton *nextBtn, *prevBtn, *starBtn, *navBtn, *closeBtn, *minBtn;
	NOWImage *playbackPanel, *infoPanel, *winampIcon, *aboutPanel, *settingsLogo;
	NOWImageWithAnalyser *albumArt;
	NOWSlider *volSlider;
	NOWGhostSlider *seekBar;
	NOWToggleButton *muteBtn;
	NOWLabel *artistLabel;
	NOWTimeLabel *elapsedTimeLabel,*totalTimeLabel;
	KFont *normalFont,*headerFont,*smallFont,*tooSmallFont,*tickerFont;
	NOWCheckBox *agcCheckBox,*hifiCheckBox,*visCheckBox,*repeatCheckBox,*customColorCheckBox;
	NOWPlayButtton *playBtn;
	NOWImageLabel *bitrateLabel,*stereoLabel,*srateLabel;
	NOWScrollLabel *tickerLabel;
	NOWSongTitleLabel *titleLabel;
	NOWRotatingImage *rotatingFlower;
	int starHideDelay,upperStarPos,upperStarStepIndex;
	int downStarPos;
	bool showFirstHalf;
	LICE_IBitmap *starsImg;

	volatile int blurAmount;

	COLORREF customColor;
	int colorThemeIndex;
	int currentlyActivePanel;

	HSTREAM hstream;
	HWADSP hwadsp;

	KTimer seekBarTimer;

	WIN32_FIND_DATAW findData;
	HANDLE hfind;
	KString searchString,currentPlayingDir;
	KPointerList<KString*> *playList;
	int currentTrackIndex;
public:

	void BlendTopEdges()
	{
		// top left edge
		for (int i = 8; i<20; i++)
		{
			for (int j = 7; j<19; j++)
			{
				LICE_pixel p = LICE_GetPixel(backPanel, j, i);
				LICE_pixel p2 = LICE_GetPixel(shapeImg, j, i);
				if (LICE_GETA(p2)>50)
					p = LICE_RGBA(LICE_GETR(p), LICE_GETG(p), LICE_GETB(p), LICE_GETA(p2));
				else
					p &= LICE_RGBA(255, 255, 255, 0);
				LICE_PutPixel(backPanel, j, i, p, 1.0f, 0);
			}
		}

		// top right edge
		for (int i = 8; i<20; i++)
		{
			for (int j = 318; j<330; j++)
			{
				LICE_pixel p = LICE_GetPixel(backPanel, j, i);
				LICE_pixel p2 = LICE_GetPixel(shapeImg, j, i);
				if (LICE_GETA(p2)>50)
					p = LICE_RGBA(LICE_GETR(p), LICE_GETG(p), LICE_GETB(p), LICE_GETA(p2));
				else
					p &= LICE_RGBA(255, 255, 255, 0);
				LICE_PutPixel(backPanel, j, i, p, 1.0f, 0);
			}
		}

	}

	void DrawOSCLines()
	{
		if (currentlyActivePanel != 0)
		{

			if (visCheckBox->IsToggled() && (hstream != 0) && (BASS_ChannelIsActive(hstream) == BASS_ACTIVE_PLAYING))
			{
				// code from bass library

				#define SPECWIDTH 293
				#define SPECHEIGHT 50
				#define SPEC_X 24
				#define SPEC_Y 97

				int x, y;
				int c;
				float *buf;
				BASS_CHANNELINFO ci;
				BASS_ChannelGetInfo(hstream, &ci); // get number of channels
				buf = (float*)alloca(ci.chans*SPECWIDTH*sizeof(float)); // allocate buffer for data
				BASS_ChannelGetData(hstream, buf, (ci.chans*SPECWIDTH*sizeof(float)) | BASS_DATA_FLOAT); // get the sample data (floating-point to avoid 8 & 16 bit processing)
				for (c = 0; c < ci.chans; c++) {
					for (x = 0; x < SPECWIDTH; x++) {
						int v = (1 - buf[x*ci.chans + c])*SPECHEIGHT / 2; // invert and scale to fit display
						if (v < 0) v = 0;
						else if (v >= SPECHEIGHT) v = SPECHEIGHT - 1;
						if (!x) y = v;
						do { // draw line from previous sample...
							if (y<v) y++;
							else if (y>v) y--;
							//specbuf[y*SPECWIDTH + x] = c & 1 ? 127 : 1; // left=green, right=red (could add more colours to palette for more chans)
							LICE_PutPixel(backPanel, SPEC_X + x, SPEC_Y + y, LICE_RGBA(255, 255, 255, 255), 1.0f, 0);
						} while (y != v);
					}
				}

			}
		}
	}

	void DrawWindow()
	{
		// clear back buffer and draw window shape image
		//LICE_FillRect(backBuffer, 0, 0, IMG_WIDTH, IMG_HEIGHT, LICE_RGBA(255, 255, 255, 0), 0.0F, LICE_BLIT_MODE_COPY);
		LICE_Blit(backBuffer, shapeImg, 0, 0, 0, 0, IMG_WIDTH, IMG_HEIGHT, 1.0, LICE_BLIT_USE_ALPHA);
		
		// draw grad img on back panel
		LICE_Blit(backPanel, gradImage, 0, 0, 0, 0, IMG_WIDTH, IMG_HEIGHT, 1.0, LICE_BLIT_MODE_COPY);

		// draw balls on back panel
		for (int i = 0; i<PARTICLE_COUNT; i++)
		{
			float prevX = particles[i]->x;
			float prevY = particles[i]->y;
			particles[i]->updateParticle();
			if (particles[i]->size)
			{
				LICE_Blit(backPanel, ballImage, roundf((particles[i]->x + prevX) / (float)2), roundf((particles[i]->y + prevY) / (float)2), 0, 24 + (particles[i]->size * 24), 24, 24, particles[i]->alpha, LICE_BLIT_USE_ALPHA);
			}
		}

		// draw lines on back panel
		LICE_Blit(backPanel, linesImage, 0, (IMG_HEIGHT - 200) / 2, lineCurrentXPos, 0, IMG_WIDTH, 200, 1.0f, LICE_BLIT_USE_ALPHA);
		lineCurrentXPos+=2;
		if (lineCurrentXPos > 943)
			lineCurrentXPos = 0;

		// draw osc lines
		DrawOSCLines();

		// draw ui components on back panel	
		for (int i = 0; i < componentList->GetSize(); i++)
		{
			NOWComponent* component = componentList->GetPointer(i);
			if (component->IsVisible())
				component->OnPaint(backPanel);
		}

		// blur if navigated
		if (blurAmount)
		{
			for (int i = 0; i < blurAmount;i++)
				LICE_Blur(backPanel, backPanel, 27, 49, 27, 49, 290, 147);
		
			blurAmount -= 1;

			if (blurAmount<0)
				blurAmount = 0;
		}


		// draw stars
		if (starHideDelay>0)
		{
			starHideDelay--;
		}else // show upper & down star
		{
			int starPattern[96] = { -1, 6, 5, 4, 3, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 2, 2, 1, 1, 0, 0, 0, 1, 1, 2, 2, 4, 5, 6, -1,
				-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, 
				-1, -1, -1, -1, -1, -1, -1,-1, -1, 6, 5, 4, 3, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 2, 2, 1, 1, 0, 0, 0, 1, 1, 2, 2, 4, 5, 6, -1 };

			if ((showFirstHalf & (upperStarStepIndex < 48)) || ((!showFirstHalf)&(48 < upperStarStepIndex)))
			{

				if (starPattern[upperStarStepIndex] != -1) // -1 means do not show
					LICE_Blit(backPanel, starsImg, upperStarPos, 10, 0, 17 + (starPattern[upperStarStepIndex] * 70), 68, 70, 1.0, LICE_BLIT_USE_ALPHA);


				if (starPattern[upperStarStepIndex] != -1) // -1 means do not show
					LICE_Blit(backPanel, starsImg, downStarPos, 170, 0, 17 + (starPattern[upperStarStepIndex] * 70), 68, 70, 1.0, LICE_BLIT_USE_ALPHA);
			}

			upperStarPos += 3;
			upperStarStepIndex++;

			downStarPos -= 3;

			if (upperStarStepIndex == 96) // star came to end
			{
				upperStarPos = -5;
				upperStarStepIndex = 0;

				downStarPos = 280;

				starHideDelay = 100 + (GetRandomNumber() % 100 * 2);
				showFirstHalf = (GetRandomNumber() % 2)==1;
			}
		}

		// draw glass panel on back panel
		LICE_Blit(backPanel, glassImg, 0, 0, 0, 0, IMG_WIDTH, IMG_HEIGHT, 1.0, LICE_BLIT_USE_ALPHA);

		// draw mask image on back panel and clear pink area
		LICE_Blit(backPanel, maskImg, 0, 0, 0, 0, IMG_WIDTH, IMG_HEIGHT, 1.0, LICE_BLIT_USE_ALPHA);
		LICE_SetAlphaFromColorMask(backPanel, LICE_RGBA(255, 0, 255, 0));

		// process top edges
		BlendTopEdges();

		// draw back panel on backbuffer
		LICE_Blit(backBuffer, backPanel, 0, 0, 0, 0, IMG_WIDTH, IMG_HEIGHT, 1.0, LICE_BLIT_USE_ALPHA);

		// Create memory DC
		HDC hdcScreen = GetDC(NULL);
		HDC hDC = CreateCompatibleDC(hdcScreen);

		// Create memory bitmap
		HBITMAP hBmp = CreateCompatibleBitmap(hdcScreen, IMG_WIDTH, IMG_HEIGHT);
		HBITMAP hBmpOld = (HBITMAP)SelectObject(hDC, hBmp);

		BitBlt(hDC, 0, 0, compWidth, compHeight, backBuffer->getDC(), 0, 0, SRCCOPY);	

		// Call UpdateLayeredWindow
		BLENDFUNCTION blend = { 0 };
		blend.BlendOp = AC_SRC_OVER;
		blend.SourceConstantAlpha = 255;
		blend.AlphaFormat = AC_SRC_ALPHA;
		SIZE szWnd = { IMG_WIDTH, IMG_HEIGHT };
		POINT ptSrc = { 0, 0 };
		UpdateLayeredWindow(compHWND, hdcScreen, NULL, &szWnd, hDC, &ptSrc, 0, &blend, ULW_ALPHA);

		SelectObject(hDC, hBmpOld);
		DeleteObject(hBmp);
		DeleteDC(hDC);
		ReleaseDC(NULL, hdcScreen);
	}

	void Run() // drawing thread
	{
		
		while (!threadShouldStop)
		{
			LARGE_INTEGER li;
			QueryPerformanceCounter(&li);
			__int64 counterStart = li.QuadPart;

			if (enableDrawing)
				DrawWindow();

			QueryPerformanceCounter(&li);
			double deltaTime = double(li.QuadPart - counterStart) / pcFreq;

			/*char buff[256];
			sprintf(buff, "%d", (int)deltaTime);
			MessageBoxA(0, buff, buff, 0);*/

			if ((FRAME_TIME - deltaTime)>0)
				Sleep(FRAME_TIME - deltaTime);


		}
		isThreadRunning = false;
	}

	void GenerateGradImage(int themeIndex, bool isCustomColor = false, COLORREF color = 0x000000)
	{
		int h, s, v;

		if (isCustomColor)
			LICE_RGB2HSV(GetRValue(color), GetGValue(color), GetBValue(color), &h, &s, &v);
		else
			LICE_RGB2HSV(colorThemes[themeIndex][0], colorThemes[themeIndex][1], colorThemes[themeIndex][2], &h, &s, &v);

		for (int i = 0; i<IMG_HEIGHT; i++)
		{
			int newV = (i / (float)IMG_HEIGHT)*(float)v;
			int r, g, b;
			LICE_HSV2RGB(h, s, newV, &r, &g, &b);
			LICE_Line(gradImage, 0, i, IMG_WIDTH, i, LICE_RGBA(r, g, b, 255), 1.0, LICE_BLIT_MODE_COPY, true);
		}
	}

	void Panel1Visibility(bool visible)
	{
		albumArt->SetVisible(visible);
		winampIcon->SetVisible(visible);
		muteBtn->SetVisible(visible);
		volSlider->SetVisible(visible);
		seekBar->SetVisible(visible);
		titleLabel->SetVisible(visible);
		artistLabel->SetVisible(visible);
		elapsedTimeLabel->SetVisible(visible);
		totalTimeLabel->SetVisible(visible);
		stereoLabel->SetVisible(visible);
		bitrateLabel->SetVisible(visible);
		srateLabel->SetVisible(visible);
	}

	void Panel2Visibility(bool visible)
	{
		settingsLogo->SetVisible(visible);
		agcCheckBox->SetVisible(visible);
		hifiCheckBox->SetVisible(visible);
		visCheckBox->SetVisible(visible);
		repeatCheckBox->SetVisible(visible);
		customColorCheckBox->SetVisible(visible);
	}

	void Panel3Visibility(bool visible)
	{
		tickerLabel->SetVisible(visible);
		aboutPanel->SetVisible(visible);
	}

	void set_config_int(const wchar_t *name, int val) {
		wchar_t str[40];
		wsprintfW(str, L"%d", val);
		WritePrivateProfileStringW(L"now_player", name, str, szConfPath);
	}

	int get_config_int(const wchar_t *name, int def = 0) {
		wchar_t tmp[40], tmp2[40];
		wsprintfW(tmp, L"%d", def);
		wsprintfW(tmp2, L"%d", def);
		GetPrivateProfileStringW(L"now_player", name, tmp2, tmp, 39, szConfPath);
		return _wtoi(tmp);
	}

	void init(wchar_t* appPath)
	{
		LARGE_INTEGER li;
		QueryPerformanceFrequency(&li);
		pcFreq = double(li.QuadPart) / 1000.0;

		applicationPath = appPath;

		enableDrawing = true;

		starHideDelay = 50 ;
		upperStarPos = -5;
		upperStarStepIndex = 0;
		downStarPos = 280;
		showFirstHalf = 1;

		lineCurrentXPos = 0;

		colorThemeIndex = 0;

		currentlyActivePanel = 0;

		blurAmount = 0;

		hstream = 0;
		hwadsp = 0;

		hfind = INVALID_HANDLE_VALUE;
		playList = new KPointerList<KString*>();
		currentTrackIndex = -1;

		// initialize drag data!
		windowDraging = false;

		componentList = new KPointerList<NOWComponent*>();
		mouseOverComponent = 0;
		mouseFocusedComponent = 0;

		// get settings dir
		SHGetFolderPathW(NULL, CSIDL_LOCAL_APPDATA, NULL, 0, szConfPath);
		wcscat(szConfPath, L"\\CrownSoft");
		CreateDirectoryW(szConfPath, 0);
		wcscat(szConfPath, L"\\NOW Player");
		CreateDirectoryW(szConfPath, 0);
		wcscat(szConfPath, L"\\config.ini");

		backBuffer = new LICE_SysBitmap(IMG_WIDTH, IMG_HEIGHT);
		backPanel = new LICE_SysBitmap(IMG_WIDTH, IMG_HEIGHT);
		gradImage = new LICE_MemBitmap(IMG_WIDTH, IMG_HEIGHT);

		HINSTANCE hInstance = KPlatformUtil::GetInstance()->GetAppInstance();
		ballImage = LICE_LoadPNGFromResource(hInstance, IDB_PNG1);
		linesImage = LICE_LoadPNGFromResource(hInstance, IDB_PNG2);

		shapeImg = LICE_LoadPNGFromResource(hInstance, IDB_PNG3);
		maskImg = LICE_LoadPNGFromResource(hInstance, IDB_PNG4);
		glassImg = LICE_LoadPNGFromResource(hInstance, IDB_PNG5);
		starsImg = LICE_LoadPNGFromResource(hInstance, IDB_PNG6);

		normalFont = new KFont(L"Tahoma", 15);
		headerFont = new KFont(L"Tahoma", 15, true);
		smallFont = new KFont(L"Tahoma", 12);
		tooSmallFont = new KFont(L"Arial", 10,false,false,true);
		tickerFont = new KFont(L"Arial", 14);

		for (int i = 0; i<PARTICLE_COUNT; i++)
		{
			particles[i] = new Particle();
		}

		playbackPanel = new NOWImage();
		playbackPanel->SetImage(hInstance, IDB_PNG7);
		playbackPanel->SetPosition((IMG_WIDTH - 210) / 2, 193);
		AddNowComponent(playbackPanel);

		infoPanel = new NOWImage();
		infoPanel->SetImage(hInstance, IDB_PNG8);
		infoPanel->SetPosition((IMG_WIDTH - 328) / 2, 23);
		AddNowComponent(infoPanel);

		rotatingFlower = new NOWRotatingImage(&hstream);
		rotatingFlower->SetImage(hInstance, IDB_PNG34);
		rotatingFlower->SetPosition(30, 219);
		AddNowComponent(rotatingFlower);

		albumArt = new NOWImageWithAnalyser();
		albumArt->SetImage(hInstance, IDB_PNG9);
		albumArt->SetPosition(36, 57);
		AddNowComponent(albumArt);

		winampIcon = new NOWImage();
		winampIcon->SetImage(hInstance, IDB_PNG10);
		winampIcon->SetPosition(125, 87);
		AddNowComponent(winampIcon);

		navBtn = new NOWButton(false);
		navBtn->SetImages(hInstance, IDB_PNG11, IDB_PNG12);
		navBtn->SetPosition(296, 95);
		navBtn->SetListener(this);
		AddNowComponent(navBtn);

		playBtn = new NOWPlayButtton();
		playBtn->SetImages(hInstance, IDB_PNG13, IDB_PNG14, IDB_PNG15, IDB_PNG16);
		playBtn->SetPosition((IMG_WIDTH - 42) / 2, 218);
		playBtn->SetListener(this);
		AddNowComponent(playBtn);
		
		nextBtn = new NOWButton();
		nextBtn->SetImages(hInstance, IDB_PNG17, IDB_PNG18);
		nextBtn->SetPosition(((IMG_WIDTH - 42) / 2)+46, 218);
		nextBtn->SetListener(this);
		AddNowComponent(nextBtn);

		prevBtn = new NOWButton();
		prevBtn->SetImages(hInstance, IDB_PNG19, IDB_PNG20);
		prevBtn->SetPosition(((IMG_WIDTH - 42) / 2) - 46, 218);
		prevBtn->SetListener(this);
		AddNowComponent(prevBtn);

		starBtn = new NOWButton();
		starBtn->SetImages(hInstance, IDB_PNG21, IDB_PNG22);
		starBtn->SetPosition(((IMG_WIDTH - 42) / 2) + 120, 219);
		starBtn->SetListener(this);
		AddNowComponent(starBtn);

		closeBtn = new NOWButton(false);
		closeBtn->SetImages(hInstance, IDB_PNG23, IDB_PNG24);
		closeBtn->SetPosition(301, 16);
		closeBtn->SetListener(this);
		AddNowComponent(closeBtn);

		minBtn = new NOWButton(false);
		minBtn->SetImages(hInstance, IDB_PNG25, IDB_PNG26);
		minBtn->SetPosition(282, 16);
		minBtn->SetListener(this);
		AddNowComponent(minBtn);

		muteBtn = new NOWToggleButton();
		muteBtn->SetImages(hInstance, IDB_PNG27, IDB_PNG28);
		muteBtn->SetPosition(127, 118);
		muteBtn->SetListener(this);
		AddNowComponent(muteBtn);

		volSlider = new NOWSlider();
		volSlider->SetSize(100, 16);
		volSlider->SetPosition(151, 119);
		volSlider->SetListener(this);
		AddNowComponent(volSlider);

		seekBar = new NOWGhostSlider();
		seekBar->SetSize(254, 16);
		seekBar->SetPosition(43, 176);
		seekBar->SetListener(this);
		AddNowComponent(seekBar);

		titleLabel = new NOWSongTitleLabel(headerFont);
		titleLabel->SetSize(169, 20);
		titleLabel->SetPosition(127, 63);
		titleLabel->SetText(L"Track Title");
		AddNowComponent(titleLabel);

		artistLabel = new NOWLabel(normalFont);
		artistLabel->SetText(L"Artist Name");
		artistLabel->SetSize(148, 20);
		artistLabel->SetPosition(150, 87);
		AddNowComponent(artistLabel);

		elapsedTimeLabel = new NOWTimeLabel(smallFont);
		elapsedTimeLabel->SetTime(0, 0, 0);
		elapsedTimeLabel->SetPosition(45, 160);
		AddNowComponent(elapsedTimeLabel);

		totalTimeLabel = new NOWTimeLabel(smallFont,false);
		totalTimeLabel->SetTime(0, 0, 0);
		totalTimeLabel->SetPosition(251, 160);
		AddNowComponent(totalTimeLabel);

		stereoLabel = new NOWImageLabel(tooSmallFont);
		stereoLabel->SetText(L"Stereo");
		stereoLabel->SetImage(hInstance, IDB_PNG29);
		stereoLabel->SetPosition(96, 163);
		AddNowComponent(stereoLabel);

		bitrateLabel = new NOWImageLabel(tooSmallFont);
		bitrateLabel->SetText(L"320 kbps");
		bitrateLabel->SetImage(hInstance, IDB_PNG29);
		bitrateLabel->SetPosition(148, 163);
		AddNowComponent(bitrateLabel);

		srateLabel = new NOWImageLabel(tooSmallFont);
		srateLabel->SetText(L"44100K");
		srateLabel->SetImage(hInstance, IDB_PNG29);
		srateLabel->SetPosition(200, 163);
		AddNowComponent(srateLabel);

		settingsLogo = new NOWImage();
		settingsLogo->SetImage(hInstance, IDB_PNG33);
		settingsLogo->SetPosition(195, 95);
		AddNowComponent(settingsLogo);

		agcCheckBox = new NOWCheckBox(normalFont);
		agcCheckBox->SetImages(hInstance, IDB_PNG30, IDB_PNG31);
		agcCheckBox->SetText(L"Enable Automatic Gain Control"); // only call this method after setting images!
		agcCheckBox->SetPosition(40, 67);
		agcCheckBox->SetListener(this);
		AddNowComponent(agcCheckBox);

		hifiCheckBox = new NOWCheckBox(normalFont);
		hifiCheckBox->SetImages(hInstance, IDB_PNG30, IDB_PNG31);
		hifiCheckBox->SetText(L"Enable Hi-Fi Audio Engine"); // only call this method after setting images!
		hifiCheckBox->SetPosition(40, 90);
		hifiCheckBox->SetListener(this);
		AddNowComponent(hifiCheckBox);

		visCheckBox = new NOWCheckBox(normalFont);
		visCheckBox->SetImages(hInstance, IDB_PNG30, IDB_PNG31);
		visCheckBox->SetText(L"Show Spectrum Analyser"); // only call this method after setting images!
		visCheckBox->SetPosition(40, 113);
		visCheckBox->SetListener(this);
		AddNowComponent(visCheckBox);
		
		customColorCheckBox = new NOWCheckBox(normalFont);
		customColorCheckBox->SetImages(hInstance, IDB_PNG30, IDB_PNG31);
		customColorCheckBox->SetText(L"Use Custom Color For Theme"); // only call this method after setting images!
		customColorCheckBox->SetPosition(40, 136);
		customColorCheckBox->SetListener(this);
		AddNowComponent(customColorCheckBox);

		repeatCheckBox = new NOWCheckBox(normalFont);
		repeatCheckBox->SetImages(hInstance, IDB_PNG30, IDB_PNG31);
		repeatCheckBox->SetText(L"Repeat Playback"); // only call this method after setting images!
		repeatCheckBox->SetPosition(40, 159);
		repeatCheckBox->SetListener(this);
		AddNowComponent(repeatCheckBox);

		aboutPanel = new NOWImage();
		aboutPanel->SetImage(hInstance, IDB_PNG32);
		aboutPanel->SetPosition(41, 59);
		AddNowComponent(aboutPanel);

		tickerLabel = new NOWScrollLabel(tickerFont);
		tickerLabel->SetPosition(41, 170);
		AddNowComponent(tickerLabel);

		Panel2Visibility(false);
		Panel3Visibility(false);

		wndIcon.LoadFromResource(IDI_ICON1);

		SetText(L"NOW-Player");
		SetStyle(WS_POPUP | WS_SYSMENU | WS_MINIMIZEBOX);
		SetExStyle(WS_EX_LAYERED);
		SetSize(IMG_WIDTH, IMG_HEIGHT);
		CreateComponent();
		SetIcon(&wndIcon);
		DrawWindow();
		CenterScreen();
		DragAcceptFiles(compHWND, TRUE);

		BASS_Init(-1, 44100, 0, compHWND, NULL);
		BASS_WADSP_Init(compHWND);
		KString dspPath = applicationPath + L"\\hifi-audio.dll";
		LoadWADSPFile((char*)(const char*)dspPath);
		TAGS_SetUTF8(TRUE);

		// apply settings
		volSlider->SetValue(get_config_int(L"vol", 50));
		BASS_SetConfig(BASS_CONFIG_GVOL_STREAM, (volSlider->GetValue() / (float)100) * 10000);
		agcCheckBox->SetToggleState(get_config_int(L"agc", 1) == 1);
		IsAGCEnabled = agcCheckBox->IsToggled();
		hifiCheckBox->SetToggleState(get_config_int(L"hifi", 0) == 1);
		visCheckBox->SetToggleState(get_config_int(L"vis", 1) == 1);
		repeatCheckBox->SetToggleState(get_config_int(L"repeat", 0) == 1);
		customColorCheckBox->SetToggleState(get_config_int(L"usecol", 0) == 1);
		customColor = get_config_int(L"col", 16711935);
		colorThemeIndex = get_config_int(L"theme", 0);
		GenerateGradImage(colorThemeIndex, customColorCheckBox->IsToggled(), customColor);

		seekBarTimer.SetTimerWindow(this);
		seekBarTimer.SetInterval(1000);
		seekBarTimer.SetListener(this);
		seekBarTimer.StartTimer();

		this->StartThread(); // start animation drawing
	}

	void OnTimer(KTimer *timer)
	{
		if (timer == &seekBarTimer)
		{
			if (hstream)
			{
				QWORD posByte = BASS_ChannelGetPosition(hstream, BASS_POS_BYTE);
				double totalSecs = BASS_ChannelBytes2Seconds(hstream, posByte);

				int hours, mins, secs;
				ConvertToTime(totalSecs, &hours, &mins, &secs);
				elapsedTimeLabel->SetTime(hours, mins, secs);

				QWORD length = BASS_ChannelGetLength(hstream, BASS_POS_BYTE);

				seekBar->SetValue((posByte / (float)length) * 100);

				if (seekBar->GetValue() == 100)
				{
					albumArt->SetEnableVis(false, hstream);
					playBtn->SetPlayingState(false);
					seekBar->SetValue(0);

					if (repeatCheckBox->IsToggled())
						PlayFile();
					else
						OnButtonPress(nextBtn);
				}
			}
		}
	}

	void OnSliderChange(void* slider)
	{
		if (slider == volSlider)
		{
			BASS_SetConfig(BASS_CONFIG_GVOL_STREAM, (volSlider->GetValue()/(float)100)*10000);
			muteBtn->SetToggleState(false);
		}
		else if (slider == seekBar)
		{
			if (hstream)
			{
				QWORD length = BASS_ChannelGetLength(hstream, BASS_POS_BYTE);
				BASS_ChannelSetPosition(hstream, (seekBar->GetValue() / (float)100)*length, BASS_POS_BYTE);

				// update elapsed time lable on same time
				QWORD posByte = BASS_ChannelGetPosition(hstream, BASS_POS_BYTE);
				double totalSecs = BASS_ChannelBytes2Seconds(hstream, posByte);

				int hours, mins, secs;
				ConvertToTime(totalSecs, &hours, &mins, &secs);
				elapsedTimeLabel->SetTime(hours, mins, secs);
			}
		}
	}

	void SetMouseFocusTo(NOWComponent *component)
	{
		mouseFocusedComponent = component;
	}

	void OnButtonPress(void* button)
	{
		if (button == starBtn)
		{
			if (customColorCheckBox->IsToggled())
			{
				CHOOSECOLORW color;
				COLORREF ccref[16];

				memset(&color, 0, sizeof(CHOOSECOLORW));
				memset(&ccref, 0, sizeof(COLORREF)*16);

				color.lStructSize = sizeof(CHOOSECOLORW);
				color.hwndOwner = compHWND;
				color.rgbResult = customColor;
				color.lpCustColors = ccref;
				color.Flags = CC_RGBINIT | CC_FULLOPEN;

				if (ChooseColorW(&color))
				{
					customColor = color.rgbResult;
					GenerateGradImage(0, true, customColor);
				}
				
			}
			else
			{
				colorThemeIndex++;
				if (colorThemeIndex == COLOR_THEMES_COUNT)
					colorThemeIndex = 0;

				// generate grad img
				GenerateGradImage(colorThemeIndex);
			}
		}
		else if (button == navBtn)
		{
			blurAmount = 4;

			if (currentlyActivePanel == 0)
			{
				Panel1Visibility(false);
				Panel2Visibility(true);
				Panel3Visibility(false);

				currentlyActivePanel++;
			}
			else if (currentlyActivePanel == 1)
			{
				Panel1Visibility(false);
				Panel2Visibility(false);
				Panel3Visibility(true);

				currentlyActivePanel++;
			}
			else if (currentlyActivePanel == 2)
			{
				Panel1Visibility(true);
				Panel2Visibility(false);
				Panel3Visibility(false);

				currentlyActivePanel = 0;
			}

		}
		else if (button == playBtn)
		{
			if (hstream)
			{
				albumArt->SetEnableVis(visCheckBox->IsToggled(), hstream);

				DWORD state = BASS_ChannelIsActive(hstream);
				if (state == BASS_ACTIVE_PLAYING)
					BASS_ChannelPause(hstream);
				else if (state == BASS_ACTIVE_PAUSED)
					BASS_ChannelPlay(hstream, FALSE);
				else if (state == BASS_ACTIVE_STOPPED)
					BASS_ChannelPlay(hstream, TRUE);
			}
		}
		else if (button == agcCheckBox)
		{
			IsAGCEnabled = agcCheckBox->IsToggled(); 
		}
		else if (button == hifiCheckBox)
		{
			EnableWADSP(hifiCheckBox->IsToggled());
		}
		else if (button == customColorCheckBox)
		{
			GenerateGradImage(colorThemeIndex, customColorCheckBox->IsToggled(), customColor);
		}
		else if (button == muteBtn)
		{
			if (muteBtn->IsToggled())
				BASS_SetConfig(BASS_CONFIG_GVOL_STREAM, 0);
			else
				BASS_SetConfig(BASS_CONFIG_GVOL_STREAM, (volSlider->GetValue() / (float)100) * 10000);
		}
		else if (button == closeBtn)
		{
			SendMessageW(compHWND, WM_CLOSE, 0, 0);
			
		}
		else if (button == minBtn)
		{
			SendMessageW(compHWND, WM_SYSCOMMAND, SC_MINIMIZE, 0);
		}
		else if (button == nextBtn)
		{
			if (currentTrackIndex == (playList->GetSize() - 1)) // last track, query from dir to get next song
			{
				if (hfind == INVALID_HANDLE_VALUE)
				{
					hfind = FindFirstFileW(searchString, &findData);
					if (hfind != INVALID_HANDLE_VALUE)
					{
						KString filePath = currentPlayingDir + L"\\" + KString((const wchar_t*)findData.cFileName);
						playList->AddPointer(new KString(filePath));
						currentTrackIndex++;
						LoadFile(filePath);
						PlayFile();
					}
				}
				else
				{
					if (FindNextFileW(hfind, &findData))
					{
						KString filePath = currentPlayingDir + L"\\" + KString((const wchar_t*)findData.cFileName);
						playList->AddPointer(new KString(filePath));
						currentTrackIndex++;
						LoadFile(filePath);
						PlayFile();
					}
				}
			}
			else // fetch next track from playlist
			{
				currentTrackIndex++;
				KString *track = playList->GetPointer(currentTrackIndex);
				LoadFile(*track);
				PlayFile();
			}
		}
		else if (button == prevBtn)
		{
			if (currentTrackIndex > 0)
			{
				currentTrackIndex--;
				KString *track = playList->GetPointer(currentTrackIndex);
				LoadFile(*track);
				PlayFile();
			}
		}
		else if (button == visCheckBox)
		{
			albumArt->SetEnableVis(visCheckBox->IsToggled(), hstream);
		}
	}

	void OnClose()
	{
		SetVisible(false);

		seekBarTimer.StopTimer();

		this->ThreadShouldStop();

		while (isThreadRunning){} //wait till animation drawing thread finish

		BASS_Stop();

		if (hwadsp)
		{
			BASS_WADSP_Stop(hwadsp);
			BASS_WADSP_FreeDSP(hwadsp);
		}

		if (hstream)
		{
			BASS_StreamFree(hstream);
		}

		BASS_WADSP_Free();
		BASS_Free();

		ClearSearchData();
		delete playList;

		DragAcceptFiles(compHWND, FALSE);

		// save settings
		set_config_int(L"vol",volSlider->GetValue());
		set_config_int(L"agc", agcCheckBox->IsToggled()?1:0);
		set_config_int(L"hifi", hifiCheckBox->IsToggled() ? 1 : 0);
		set_config_int(L"vis", visCheckBox->IsToggled() ? 1 : 0);
		set_config_int(L"repeat", repeatCheckBox->IsToggled() ? 1 : 0);
		set_config_int(L"usecol", customColorCheckBox->IsToggled() ? 1 : 0);
		set_config_int(L"col", customColor);
		set_config_int(L"theme", colorThemeIndex);

		::DestroyWindow(compHWND);
	}

	void OnPaint() // handle random dummy paint msgs
	{
		PAINTSTRUCT ps;
		BeginPaint(compHWND, &ps);

		EndPaint(compHWND, &ps);
	}

	void AddNowComponent(NOWComponent *component)
	{
		if (component)
			componentList->AddPointer(component);
	}

	NOWComponent* GetComponentAt(int xPos, int yPos)
	{
		for (int i = componentList->GetSize()-1; i>=0; i--) // search top to bottom
		{
			NOWComponent* component = componentList->GetPointer(i);
			if (component->IsVisible())
			{
				if (component->IsMouseWithinArea(xPos, yPos))
				{
					return component;
				}
			}
		}
		return 0;
	}

	void OnMouseLDown(int xPos, int yPos)
	{
		NOWComponent* component = GetComponentAt(xPos, yPos);

		if (component) // LDown on component!
		{
			component->OnMouseLDown(xPos, yPos); // send event!
		}
		else // LDown for client area drag
		{
			OnMLButtonDownForDrag(xPos, yPos);
		}
	}

	void OnMouseMove(int xPos, int yPos, WPARAM fwKeys)
	{
		if (windowDraging) // client area drag!
		{
			OnMouseMoveForDrag();
		}
		else // send event to appropriate component!
		{

			if (mouseFocusedComponent)
			{
				mouseFocusedComponent->OnMouseMove(xPos, yPos, fwKeys);
				return;
			}

			NOWComponent* component = GetComponentAt(xPos, yPos);

			if (component)// mouse is on component!
			{
				if (component == mouseOverComponent)// mouse is still on prev comp!
				{
					component->OnMouseMove(xPos, yPos, fwKeys);
				}
				else // mouse is on new component!
				{
					if (mouseOverComponent)// tell it to prev comp!
						mouseOverComponent->OnMouseLost(fwKeys);

					mouseOverComponent = component;// new mouse over comp!
					mouseOverComponent->OnMouseOver(fwKeys);
				}
			}
			else // mouse is not on component!
			{
				if (mouseOverComponent)// tell it to prev comp!
				{
					mouseOverComponent->OnMouseLost(fwKeys);
					mouseOverComponent = 0;
				}
			}


		}
	}

	void OnMouseLUp(int xPos, int yPos)
	{
		if (windowDraging) // client area drag!
		{
			OnMLButtonUpForDrag();
		}
		else // send event to appropriate component!
		{

			if (mouseFocusedComponent)
			{
				mouseFocusedComponent->OnMouseLUp(xPos, yPos);
				return;
			}

			NOWComponent* component = GetComponentAt(xPos, yPos);

			if (component)// mouse is on component!
			{
				component->OnMouseLUp(xPos, yPos);
			}

		}
	}

	void OnMouseRDown(int xPos, int yPos)
	{
		NOWComponent* component = GetComponentAt(xPos, yPos);
		if (component)
			component->OnMouseRDown(xPos, yPos);
	}

	void OnMouseRUp(int xPos, int yPos)
	{
		NOWComponent* component = GetComponentAt(xPos, yPos);
		if (component)
			component->OnMouseRUp(xPos, yPos);
	}

	void ConvertToTime(int totalSeconds, int *hours, int *mins, int *secs)
	{
		int _hours = totalSeconds / 3600;
		*hours = _hours;
		if (hours)
		{		
			int remainingSecs = totalSeconds - (_hours * 3600);
			int _mins = remainingSecs / 60;
			*mins = _mins;
			if (_mins)
			{
				remainingSecs = remainingSecs - (_mins * 60);
				*secs = remainingSecs;
			}
			else
			{
				*secs = remainingSecs;
			}
		}
		else
		{
			int _mins = totalSeconds / 60;
			*mins = _mins;
			if (_mins)
			{
				int remainingSecs = totalSeconds - (_mins * 60);
				*secs = remainingSecs;
			}
			else
			{
				*secs = totalSeconds;
			}
		}
	}

	void LoadWADSPFile(char* file)
	{
		hwadsp = BASS_WADSP_Load(file, 0, 0, 400, 200, 0);
		if (hwadsp)
		{
			BASS_WADSP_Start(hwadsp, 0, 0);
		}
	}

	void EnableWADSP(int enable)
	{
		if (hwadsp)
		{
			if (hstream)
			{
				if (enable)
				{
					BASS_WADSP_ChannelSetDSP(hwadsp, hstream, 1);
				}
				else
				{
					BASS_WADSP_ChannelRemoveDSP(hwadsp);
				}
			}
		}
	}

	void LoadFile(KString &filePath)
	{
		playBtn->SetPlayingState(false);
		albumArt->SetEnableVis(false, hstream);

		// free old one
		if (hstream)
		{
			BASS_ChannelStop(hstream);
			BASS_StreamFree(hstream);
		}

		if (hfind == INVALID_HANDLE_VALUE)
		{
			// setup search path...
			wchar_t *path = (wchar_t*)malloc((filePath.GetLength() + 1)*sizeof(wchar_t));
			wcscpy(path, (const wchar_t*)filePath);

			wchar_t *p;
			for (p = path; *p; p++) {}	// find end
			for (; p > path && *p != L'\\'; p--) {} // back up to last backslash
			*p = 0;	// kill it

			currentPlayingDir = path;

			wcscat(path, L"\\*.mp3");
			searchString = path;
			free(path);
		}

		hstream = BASS_StreamCreateFile(FALSE, (const wchar_t*)filePath, 0, 0, BASS_UNICODE);

		if (hstream) // read tags
		{
			char *title = (char*)TAGS_Read(hstream, "%TITL"); // do not free returned buffer
			KString titleText = KString((const char*)title, CP_UTF8);
			if (titleText.GetLength())
			{
				char *album = (char*)TAGS_Read(hstream, "%ALBM"); // do not free returned buffer
				KString albumText = KString((const char*)album, CP_UTF8);
				if (albumText.GetLength())
					titleLabel->SetText(albumText+L" - "+titleText);
				else
					titleLabel->SetText(titleText);
			}
			else
			{
				// get file name...
				wchar_t *path = (wchar_t*)malloc((filePath.GetLength() + 1)*sizeof(wchar_t));
				wcscpy(path, (const wchar_t*)filePath);

				wchar_t *p;
				for (p = path; *p; p++) {}	// find end
				for (; p > path && *p != L'\\'; p--) {} // back up to last backslash
				p++;

				titleLabel->SetText(KString((const wchar_t*)p));

				free(path);
			}

			char *artist = (char*)TAGS_Read(hstream, "%ARTI"); // do not free returned buffer
			artistLabel->SetText(KString((const char*)artist, CP_UTF8));

			float bitrate = 0;
			BASS_ChannelGetAttribute(hstream, BASS_ATTRIB_BITRATE, &bitrate);
			bitrateLabel->SetText(KString((int)bitrate, 10) + L" kbps");

			float srate = 0;
			BASS_ChannelGetAttribute(hstream, BASS_ATTRIB_FREQ, &srate);
			srateLabel->SetText(KString((int)srate, 10) + L"K");

			QWORD length = BASS_ChannelGetLength(hstream, BASS_POS_BYTE);
			double totalSecs = BASS_ChannelBytes2Seconds(hstream, length);

			int hours, mins, secs;
			ConvertToTime(totalSecs, &hours, &mins, &secs);
			totalTimeLabel->SetTime(hours, mins, secs);

			BASS_ChannelSetDSP(hstream,AGCDSPCallback,0,0);
			EnableWADSP(hifiCheckBox->IsToggled());
		}

	}

	void PlayFile()
	{
		if (hstream)
		{	
			playBtn->SetPlayingState(true);
			BASS_ChannelPlay(hstream, TRUE);

			albumArt->SetEnableVis(visCheckBox->IsToggled(), hstream);
		}
	}

	void ClearSearchData()
	{
		if (hfind != INVALID_HANDLE_VALUE)
			FindClose(hfind);

		playList->DeleteAll();
		currentTrackIndex = -1;

		hfind = INVALID_HANDLE_VALUE;
	}

	void OnDropFile(WPARAM wParam, LPARAM lParam)
	{
		HDROP hdrop = (HDROP)wParam;

		int charCount = DragQueryFileW(hdrop, 0, NULL, 0); // get the char count to hold file path

		wchar_t *szFile = (wchar_t*)malloc((charCount + 1) * sizeof(wchar_t));

		if (DragQueryFileW(hdrop, 0, szFile, charCount+1) > 0) // get the first file only
		{
			ClearSearchData();
			LoadFile(KString(szFile));
			PlayFile();
		}

		free(szFile);

		DragFinish(hdrop);
	}

	void ChangeVolumeByStep(bool isIncrement)
	{
		#define VOL_CHANGE_AMOUNT 10

		if (isIncrement)
			volSlider->SetValue(volSlider->GetValue() + VOL_CHANGE_AMOUNT);
		else
			volSlider->SetValue(volSlider->GetValue() - VOL_CHANGE_AMOUNT);
		BASS_SetConfig(BASS_CONFIG_GVOL_STREAM, (volSlider->GetValue() / (float)100) * 10000);
		muteBtn->SetToggleState(false);
	}

	void ChangeSongPosByStep(bool isIncrement)
	{
		if (hstream)
		{
			QWORD length = BASS_ChannelGetLength(hstream, BASS_POS_BYTE);
			QWORD posByte = BASS_ChannelGetPosition(hstream, BASS_POS_BYTE);

			#define POS_CHANGE_AMOUNT 1024 * 128 * 10

			if (isIncrement)
			{
				posByte += POS_CHANGE_AMOUNT;
				if (posByte > length)
					posByte = length;
			}
			else
			{			
				if (posByte > POS_CHANGE_AMOUNT)
					posByte -= POS_CHANGE_AMOUNT;
				else
					posByte = 0;
			}

			BASS_ChannelSetPosition(hstream, posByte, BASS_POS_BYTE);
			seekBar->SetValue((posByte / (float)length) * 100);
		}
	}

	LRESULT WindowProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
	{
		switch (msg)
		{
		case WM_PAINT:
			this->OnPaint();
			break;
		case WM_ERASEBKGND:
			return 1;
		case WM_MOUSEMOVE:
			this->OnMouseMove(LOWORD(lParam), HIWORD(lParam), wParam);
			break;
		case WM_LBUTTONDOWN:
			this->OnMouseLDown(LOWORD(lParam), HIWORD(lParam));
			break;
		case WM_LBUTTONUP:
			this->OnMouseLUp(LOWORD(lParam), HIWORD(lParam));
			break;
		case WM_RBUTTONDOWN:
			this->OnMouseRDown(LOWORD(lParam), HIWORD(lParam));
			break;
		case WM_RBUTTONUP:
			this->OnMouseRUp(LOWORD(lParam), HIWORD(lParam));
			break;
		case WM_KEYUP:
			if (wParam == VK_SPACE)
				playBtn->OnButtonClick();
			else if (wParam == VK_UP)
				ChangeVolumeByStep(true);
			else if (wParam == VK_DOWN)
				ChangeVolumeByStep(false);
			else if (wParam == VK_LEFT)
				ChangeSongPosByStep(false);
			else if (wParam == VK_RIGHT)
				ChangeSongPosByStep(true);
			else if (wParam == VK_TAB)
				OnButtonPress(navBtn);
			else
				return KWindow::WindowProc(hwnd, msg, wParam, lParam);
			break;
		case WM_DROPFILES:
			this->OnDropFile(wParam, lParam);
			break;
		case WM_COPYDATA:
			{
				PCOPYDATASTRUCT pCopyData = (PCOPYDATASTRUCT)lParam;
				ClearSearchData();
				LoadFile(KString((const wchar_t*)pCopyData->lpData));
				PlayFile();
			}
			return TRUE;
		case WM_MOUSEWHEEL:
			if (GET_WHEEL_DELTA_WPARAM(wParam) > 0)
				ChangeVolumeByStep(true);
			else
				ChangeVolumeByStep(false);
			break;
		case WM_SYSCOMMAND:
			if (wParam == SC_MINIMIZE)
				enableDrawing = false;
			else if (wParam == SC_RESTORE)
				enableDrawing = true;
			return KWindow::WindowProc(hwnd, msg, wParam, lParam);
		default:
			return KWindow::WindowProc(hwnd, msg, wParam, lParam);
		}

		return 0;
	}

	void OnMLButtonDownForDrag(int xPos, int yPos)
	{
		windowDraging = true;
		SetCapture(compHWND);

		cld_mouse_x = xPos;
		cld_mouse_y = yPos;
	}

	void OnMouseMoveForDrag()
	{
		POINT pos;
		GetCursorPos(&pos);
		SetPosition(pos.x - cld_mouse_x, pos.y - cld_mouse_y);
	}

	void OnMLButtonUpForDrag()
	{
		ReleaseCapture();
		windowDraging = false;
	}

	void releaseResources()
	{
		delete backBuffer;
		delete gradImage;
		delete backPanel;
		delete shapeImg;
		delete maskImg;
		delete ballImage;
		delete glassImg;
		delete linesImage;
		delete starsImg;

		for (int i = 0; i<PARTICLE_COUNT; i++)
		{
			delete particles[i];
		}

		for (int i = 0; i < componentList->GetSize(); i++)
			delete componentList->GetPointer(i);

		delete componentList;

		delete normalFont;
		delete headerFont;
		delete smallFont;
		delete tooSmallFont;
		delete tickerFont;
	}
};

class NOWPlayer : public KApplication
{
public:
	int Main(KString **argv, int argc)
	{
		HWND oldInstance = FindWindowW(0, L"NOW-Player");
		if (oldInstance != NULL)
		{
			if (argc > 1)
			{
				KString *arg1 = argv[1];

				COPYDATASTRUCT copyData;
				copyData.dwData = 0;
				copyData.lpData = (void*)arg1->operator const wchar_t *();
				copyData.cbData = (arg1->GetLength() + 1)*sizeof(wchar_t);

				SendMessageW(oldInstance,WM_COPYDATA, 0, (LPARAM)&copyData);
			}
			return 0;
		}

		MainWindow mainWindow;

		srand(time(NULL));

		// setup application path...
		static wchar_t path[MAX_PATH];
		GetModuleFileNameW(KPlatformUtil::GetInstance()->GetAppInstance(), path, MAX_PATH);

		wchar_t *p;
		for (p = path; *p; p++) {}	// find end
		for (; p > path && *p != '\\'; p--) {} // back up to last backslash
		*p = 0;	// kill it

		mainWindow.init(path);
		mainWindow.SetVisible(true);

		if (argc > 1)
		{
			KString *arg1 = argv[1];
			if (arg1->EqualsIgnoreCase(L"-firsttime"))
			{
				KString demoPath = KString(path) + L"\\demo.mp3";
				mainWindow.LoadFile(demoPath);
			}
			else
			{
				mainWindow.LoadFile(*arg1);
			}

			mainWindow.PlayFile();
		}
		else // load last file
		{

		}

		DoMessagePump(false);

		mainWindow.releaseResources();

		return 0;
	}
};

START_RFC_APPLICATION(NOWPlayer)